package models

import (
	"github.com/jmoiron/sqlx"
	"log"
)

type ParseIpHosts struct {
	Id   int    `db:"id"`
	Host string `db:"host"`
}

type ListParseIpHosts []ParseIpHosts

func(a ListParseIpHosts) difference(b []ParseIpHosts) (diff []ParseIpHosts) {
	m := make(map[string]bool)

	for _, item := range b {
		m[item.Host] = true
	}

	for _, item := range a {
		if _, ok := m[item.Host]; !ok {
			diff = append(diff, item)
		}
	}
	return diff
}

func insertParseIpHost(ipsDiff []ParseIpHosts) []int {
	rows, err := db.NamedQuery("INSERT INTO parse_ip_hosts (host) VALUES(:host) RETURNING id", ipsDiff)
	if err != nil {
		log.Fatalln(err)
	}

	var ids []int
	for rows.Next() {
		var id int
		err := rows.Scan(&id)
		if err != nil {
			log.Fatalln(err)
		}

		ids = append(ids, id)
	}

	return ids
}

func BatchParseIpHosts(hosts []string) []int{
	var args ListParseIpHosts
	for _, host := range hosts {
		i := ParseIpHosts{}
		i.Host = host
		args = append(args, i)
	}

	hostsDb := getHavingHosts(args)
	hostsDiff := args.difference(hostsDb)

	if hostsDiff != nil{
		return insertParseIpHost(hostsDiff)
	}

	return nil
}

func getHavingHosts(hostsArg []ParseIpHosts) []ParseIpHosts{
	query, args, err := sqlx.Named("SELECT * FROM parse_ip_hosts where host IN ((:host))", hostsArg)
	if err != nil{
		log.Fatalln(err)
	}

	query, arg, err := sqlx.In(query, args...)
	if err != nil{
		log.Fatalln(err)
	}

	query = db.Rebind(query)
	rows, err := db.Queryx(query, arg...)
	if err != nil{
		log.Fatalln(err)
	}

	var hostsDb []ParseIpHosts
	for rows.Next() {
		var host ParseIpHosts
		err = rows.StructScan(&host)
		if err != nil{
			log.Fatalln(err)
		}
		hostsDb = append(hostsDb, host)
	}

	return hostsDb
}